/**
 * i-net software provides programming examples for illustration only, without warranty
 * either expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and/or fitness for a particular purpose. This programming example
 * assumes that you are familiar with the programming language being demonstrated and
 * the tools used to create and debug procedures. i-net software support professionals
 * can help explain the functionality of a particular procedure, but they will not modify
 * these examples to provide added functionality or construct procedures to meet your
 * specific needs.
 *
 * Copyright © 1999-2026 i-net software GmbH, Berlin, Germany.
**/
package cache;

import java.io.FileOutputStream;
import java.io.IOException;
import java.util.Properties;

import com.inet.report.*; //ReportException, PropertiesChecker
import com.inet.report.cache.*; //Cache, ReportCacheKey

/**
 * This file contains some sample methods, that will show the basic usage of the cache interface.
 */
public class CacheAPI {

    // a report file
    private String report = "samples/sample.rpt";

    /**
     * Main method of the sample.
     * @param args no parameters used
     * @throws ReportException if one of the called sample methods throws an exception
     */
    public static void main( String[] args ) throws ReportException {
        CacheAPI test = new CacheAPI();
        test.gettingAReportFromCacheSample();
        //test.putAReportInCacheSample();
        //test.refreshAReportInCacheSample();
        //test.showStateOfAllReportsSample();
    }

    /**
     * This sample shows, how to get the data of a report from cache. Following API calls are used here: Cache.getCache
     * Cache.getKey Cache.lock Cache.unlock Cache.getPageAndWait Cache.getTotalPageSizeAndWait Cache.getPageCountAndWait
     * @throws ReportException if any error occurred
     */
    public void gettingAReportFromCacheSample() throws ReportException {
        /*first step is creating the key of the report*/
        Cache cache = Cache.getCache();

        Properties reportProps = new Properties();//create the properties for the report
        reportProps.put( "report", report );
        reportProps.put( "export_fmt", Engine.EXPORT_PDF );
        //reportProps.put("promtAPromptFieldName","aPromptFieldValue");
        //The complete list of possible properties can be found at:
        //https://docs.inetsoftware.de/reporting/help/report-url-parameters

        // Receiving the key that identifies the report in the Cache.
        // The key contains all informations set in reportProps 
        ReportCacheKey key = cache.getKey( reportProps );//executes the report in background if not exists in Cache.
        /*last step is getting the data of the report.*/
        byte[] result = getTotalReportSafely( key );

        try {
            FileOutputStream fos = new FileOutputStream( "samples/out.pdf" );
            fos.write( result );
            fos.flush();
            fos.close();
        } catch( IOException e ) {
            e.printStackTrace();
        }
    }

    /**
     * Returns the complete data of an report from cache.
     * @param key unique key for a report
     * @return byte array containing the data of an report
     * @throws ReportException if any error occurred
     */
    private byte[] getTotalReport( ReportCacheKey key ) throws ReportException {
        //getting the cache instance
        Cache cache = Cache.getCache();
        //get the number of pages
        int pageCount = cache.getPageCountAndWait( key );
        //get the size of report
        int totalReportSize = cache.getTotalPageSizeAndWait( key );
        //create the ByteArray that stores the report
        byte[] reportData = new byte[totalReportSize];
        int offset = 0;
        //getting the data of each page
        for( int page = 1; page <= pageCount; page++ ) {
            byte[] pageData = cache.getPageAndWait( key, page );
            //put bytes in reportData
            System.arraycopy( pageData, 0, reportData, offset, pageData.length );
            offset += pageData.length;
        }
        return reportData;
    }

    /**
     * Returns the complete data of an report from Cache. The method locks the report during getting the data. While the
     * lock is set, every delete request for the report waits.
     * @param key unique key for a report
     * @return byte array containing the data of an report
     * @throws ReportException if any error occurred
     */
    private byte[] getTotalReportSafely( ReportCacheKey key ) throws ReportException {
        //getting the cache instance
        Cache cache = Cache.getCache();
        //lock the report during method runs
        cache.lock( key );
        try {
            return getTotalReport( key );
        } finally {
            //unlock must be called in every case! Otherwise its not possible to delete the report from Cache. 
            cache.unlock( key );
        }
    }

    /**
     * This sample shows, how to put a report in cache Following API calls are here used: Cache.getCache Cache.getKey
     * Cache.isFinish
     * @throws ReportException if any error occurred
     */
    public void putAReportInCacheSample() throws ReportException {
        /*first step is creating the key of the report*/
        Cache cache = Cache.getCache();
        //create the properties for the report 
        Properties reportProps = new Properties();
        reportProps.put( "report", report );
        //The complete list of possible properties can be found at:
        //https://docs.inetsoftware.de/reporting/help/report-url-parameters

        //getting the key for the report properties. Note: the getKey methods starts rendering of report, if the report was not found in in cache. During rendering, the report will be stored in the cache.
        ReportCacheKey key = cache.getKey( reportProps );
        /*last step is waiting until rendering process finishes*/
        cache.getPageCountAndWait( key );
        //now all pages of the report are finished rendered and can be accessed
    }

    /**
     * This sample shows, how to refresh a report in Cache Cache.getCache Cache.getKey Cache.createKey cache.delete
     * @throws ReportException if any error occurred
     */
    public void refreshAReportInCacheSample() throws ReportException {
        /*first step is creating the key of the report*/
        Cache cache = Cache.getCache();
        //create the properties for the report 
        Properties reportProps = new Properties();
        reportProps.put( "report", report );
        //The complete list of possible properties can be found at:
        //https://docs.inetsoftware.de/reporting/help/report-url-parameters

        //create the key for the report properties.
        ReportCacheKey key = cache.createKey( reportProps );
        /*last step is replace the report in cache*/
        if( cache.exists( key ) ) {
            cache.delete( report );//delete() expects the value of the "report" property
        }
        //add the report to cache
        cache.getKey( reportProps );
    }

    /**
     * This sample shows how to receive information about all reports known by cache.
     * @throws ReportException if any error occurred
     */
    public void showStateOfAllReportsSample() throws ReportException {
        Cache cache = Cache.getCache();
        //receive List of all reports in Cache
        System.out.println( "report \t format \t state \t writeprotected \t further parameter" );
        for( ReportCacheKey reportKey : cache.getReports() ) {
            String state = "";
            if( !cache.isFinished( reportKey ) ) {
                state = "In Execution";
            } else {
                state = cache.isFailedReport( reportKey ) ? "Finished with errors" : "Finished without errors";
            }
            String exportFormat = reportKey.getFormat();
            String report = reportKey.getReport();
            String furtherParameter = reportKey.getParameter();

            System.out.print( report + '\t' );
            System.out.print( exportFormat + '\t' );
            System.out.print( state + '\t' );
            System.out.println( furtherParameter + "\t" );
        }
    }

}
